/// \ingroup gpi
///@{

/// \file callplot.cpp
/// Call (W)GNUPLOT to run autonomously - bodies.

/// \fn void CallPlot(const String& command)
/// Call gnuplot program to run the command file.
/// \internal
/// Version depends on which define is activated.

#define WANT_STREAM
#include "include.h"

// CALLING METHOD
//#define CREATEPROCESS      // for win95 or winNT or gnuplot version 4
//#define CREATEPROCESS32      // for win95 or winNT - 32 bit gnuplot
//#define WINEXEC            // for windows 3.1
//#define SPAWNLP            // alt. for win95 or winNT
//#define SPAWNLP32          // alt. for win95 or winNT  - 32 bit gnuplot
//#define FORK               // for unix
//#define FORK_PERSIST       // for unix & X11
//#define FORK_DASH          // for unix; put gnuplot into interactive mode

#ifdef set_unix_options 
#define FORK_PERSIST       // for unix & X11
#else
#define SPAWNLP            // alt. for win95 or winNT
#endif


# ifdef CREATEPROCESS

// This version is for windows NT or windows 95

#include <windows.h>

#include "str.h"
#include "callplot.h"


#ifdef use_namespace
namespace RBD_LIBRARIES {
#endif

void CallPlot(const String& command)
{
   String command_line = ("WGNUPLOT.EXE "+command);
   char cl[100]; int sz = command_line.copy(cl, 100); cl[sz]=0;
   STARTUPINFO startupinfo =
   {
      68,0,0,0,0,0,0,0,0,0,0,STARTF_USESHOWWINDOW,SW_SHOWMINIMIZED,0,0,0,0,0
   };

   PROCESS_INFORMATION PI;
   int w = CreateProcess(
      "WGNUPLOT.EXE",
      cl,
      0,
      0,
      0,
      CREATE_SEPARATE_WOW_VDM,
      0,
      0,
      &startupinfo,
      &PI);

   if (w==0) Throw(Runtime_error("Call to gnuplot fails\n"));
}

#ifdef use_namespace
}
#endif


#endif

# ifdef CREATEPROCESS32

// This version is for windows NT or windows 95 using 32 bit version of gnuplot

#include <windows.h>

#include "str.h"
#include "callplot.h"

#ifdef use_namespace
namespace RBD_LIBRARIES {
#endif

void CallPlot(const String& command)
{
   String command_line = ("WGNUPL32.EXE "+command);
   char cl[100]; int sz = command_line.copy(cl, 100); cl[sz]=0;
   STARTUPINFO startupinfo =
   {
      68,0,0,0,0,0,0,0,0,0,0,STARTF_USESHOWWINDOW,SW_SHOWMINIMIZED,0,0,0,0,0
   };

   PROCESS_INFORMATION PI;
   int w = CreateProcess(
      "WGNUPL32.EXE",
      cl,
      0,
      0,
      0,
      0,
      0,
      0,
      &startupinfo,
      &PI);

   if (w==0) Throw(Runtime_error("Call to gnuplot fails\n"));
}

#ifdef use_namespace
}
#endif


#endif

#ifdef WINEXEC

// This version is for windows 3.1

#include <windows.h>

#include "str.h"
#include "callplot.h"

#ifdef use_namespace
namespace RBD_LIBRARIES {
#endif

void CallPlot(const String& command)
{
   int w =
      WinExec((String("WGNUPLOT ")+command).c_str(), SW_SHOWMINIMIZED);
   if (w==16)
      Throw(Runtime_error(
         "Call to gnuplot fails; one copy already running?\n"));
   if (w<32) Throw(Runtime_error("Call to gnuplot fails\n"));
}

#ifdef use_namespace
}
#endif


#endif

#ifdef SPAWNLP

// This is an alternative version for WIN32

#include <process.h>

#include "str.h"
#include "callplot.h"

#ifdef use_namespace
namespace RBD_LIBRARIES {
#endif

void CallPlot(const String& command)
{
   int w =
      spawnlp(P_NOWAIT, "WGNUPLOT", "WGNUPLOT", command.c_str(), (char*)0);
   if (w < 0) Throw(Runtime_error("Call to gnuplot fails\n"));
}

#ifdef use_namespace
}
#endif


#endif

#ifdef SPAWNLP32

// This is an alternative version for WIN32

#include <process.h>

#include "str.h"
#include "callplot.h"

#ifdef use_namespace
namespace RBD_LIBRARIES {
#endif

void CallPlot(const String& command)
{
   int w =
      spawnlp(P_NOWAIT, "WGNUPL32", "WGNUPL32", command.c_str(), (char*)0);
   if (w < 0) Throw(Runtime_error("Call to gnuplot fails\n"));
}

#ifdef use_namespace
}
#endif


#endif

#ifdef FORK

// Version for UNIX

#include "str.h"
#include "callplot.h"
#include <unistd.h>

#ifdef use_namespace
namespace RBD_LIBRARIES {
#endif

void CallPlot(const String& command)
{
   int w;
   if ((w = fork()) == 0)
   {
       execlp("gnuplot","gnuplot", command.c_str(), (char*)0);
       cout << "Call to gnuplot fails" << endl;
       exit(0);
       // but this won't stop main process
   }
}

#ifdef use_namespace
}
#endif


#endif

#ifdef FORK_PERSIST

// Version for UNIX, include persist option

#include "str.h"
#include "callplot.h"
#include <unistd.h>

#ifdef use_namespace
namespace RBD_LIBRARIES {
#endif

void CallPlot(const String& command)
{
   int w;
   if ((w = fork()) == 0)
   {
       execlp("gnuplot","gnuplot", "-persist", command.c_str(), (char*)0);
       cout << "Call to gnuplot fails" << endl;
       exit(0);
       // but this won't stop main process
   }
}

#ifdef use_namespace
}
#endif


#endif

#ifdef FORK_DASH

// This version is for Unix - put dash in command line to put gnuplot into
// interactive mode

// Version for UNIX

#include "str.h"
#include "callplot.h"
#include <unistd.h>

#ifdef use_namespace
namespace RBD_LIBRARIES {
#endif

void CallPlot(const String& command)
{
   int w;
   if ((w = fork()) == 0)
   {
       execlp("gnuplot","gnuplot", command.c_str(), " -", (char*)0);
       cout << "Call to gnuplot fails" << endl;
       exit(0);
       // but this won't stop main process
   }
}

#ifdef use_namespace
}
#endif


#endif

///@}

