/// \ingroup gpi
///@{

/// \file gnuplot.cpp
/// C++ interface to Gnuplot - bodies.
/// This stage is very low tech - write graph information to files
/// and then run gnuplot on these files.
///
/// Try to be sufficiently flexible to enable other
/// interfaces to be built.


#define WANT_STREAM
#define WANT_FSTREAM           // for file io
#define WANT_MATH

#include "gnuplot.h"

#include "newmatio.h"
#include "callplot.h"

#ifdef use_namespace
namespace RBD_LIBRARIES {
#endif



// FILE SEPARATOR
static char bs = '/';         // file name separator for unix - also windows
//static char bs[] = "\\\\";  // file name separator for msdos & windows

// EXIT OPTION select one of following
//#define PAUSE3600           // for Unix
//#define PAUSE_M1            // windows, old gnuplot
//#define PAUSE_MOUSE         // for Linux - exit on several mouse clicks
//#define DASH                // gnuplot 3.71,4 on windows
//#define SKIP                // no special pause option
//#define HOLD                // hold graphs till program terminates

#ifdef set_unix_options 
   #define PAUSE3600          // For Unix   (alternatively use SKIP)
#else
   #define DASH               // gnuplot 3.71,4 on windows
#endif




/// Make a new alphabetic name.
/// \internal
/// Used for generating a new file name. Uniqueness is guaranteed within
/// this program but not if two copies are being run simultaneously.
static String NewName()
{
   static int static_count;
   int count = static_count++;
   int c1 = count % 26;
   int c2 = (count / 26) % 26;
   int c3 = (count / (26*26)) % 26;
   char alphabet[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
   return String("GP") + alphabet[c3] + alphabet[c2] + alphabet[c1];
}

/// Throw exception if P is null.
/// \internal
/// Used for throwing an exception if a new command fails.
static void CheckSpace(const void* P) { if (!P) Throw(Bad_alloc()); }

/// Open file for holding gnuplot commands.
/// \internal
static void setup_file(String& CommandFile, ofstream& fout)
{
#ifndef HOLD
   CommandFile = TemporaryDirectory+bs+NewName()+".gp";
   fout.open(CommandFile.c_str());
#else
   if (CommandFile=="") CommandFile = TemporaryDirectory+bs+NewName()+".gp";
   fout.open(CommandFile.c_str(), ios::app);
#endif
   if (!fout) Throw(Runtime_error("File open fails\n"));
}

/// Write final gnuplot commands and execute CallPlot.
/// \internal
static void write_out(const String& CommandFile, ofstream& fout)
{
#ifdef PAUSE_M1
   fout << "pause -1" << endl;
   fout << "exit" << endl;
   fout.close();
   CallPlot(CommandFile);
#endif
#ifdef PAUSE3600
   fout << "pause 3600" << endl;
   fout << "exit" << endl;
   fout.close();
   CallPlot(CommandFile);
#endif
#ifdef PAUSE_MOUSE
   fout << "pause mouse" << endl;
   fout << "pause mouse" << endl;
   fout << "pause mouse" << endl;
   fout << "pause mouse" << endl;
   fout << "exit" << endl;
   fout.close();
   CallPlot(CommandFile);
#endif
#ifdef DASH
   fout << "exit" << endl;
   fout.close();
   CallPlot(CommandFile + " -");
#endif
#ifdef SKIP
   fout << "exit" << endl;
   fout.close();
   CallPlot(CommandFile);
#endif
#ifdef HOLD
   fout << "pause -1 \"press <enter>\"" << endl;
   fout << "reset" << endl;
   fout.close();
#endif
} 

///Main constructor for BaseSeries and derived classes.
///Create new file, copy matrix data to file and remember file name
BaseSeries::BaseSeries(const String& title, const Matrix& data)
   : Title(title)
{
   // ignore series with no elements
   if (data.Nrows() == 0) { FileName = ""; return; }

   // make file name
   FileName = NewName();

   // write data to disk
   ofstream fout((TemporaryDirectory+bs+FileName).c_str());
   if (!fout) Throw(Runtime_error("File open fails\n"));
   int nrows = data.Nrows(); int ncols = data.Ncols();
   for (int i = 1; i <= nrows; i++)
   {
      if (data(i,1) < 0.99999E50)        // assume missing otherwise
         for (int j = 1; j <= ncols; j++)
            fout << " " << setprecision(15) << setw(20) << data(i, j);
      fout << endl;
   }
   fout << endl;
}

String LineSeries::Style() const
   { String S = String("lines ") + Colour; return S; }

BaseSeries* LineSeries::Clone() const
   { BaseSeries* bs = new LineSeries(*this); CheckSpace(bs); return bs; }


String ImpulseSeries::Style() const
   { String S = String("impulses ") + Colour; return S; }

BaseSeries* ImpulseSeries::Clone() const
   { BaseSeries* bs = new ImpulseSeries(*this); CheckSpace(bs); return bs; }

String StepSeries::Style() const
   { String S = String("steps ") + Colour; return S; }

BaseSeries* StepSeries::Clone() const
   { BaseSeries* bs = new StepSeries(*this); CheckSpace(bs); return bs; }

String BoxSeries::Style() const
   { String S = String("boxes ") + Colour; return S; }

BaseSeries* BoxSeries::Clone() const
   { BaseSeries* bs = new BoxSeries(*this); CheckSpace(bs); return bs; }

String PointSeries::Style() const
   { String S = String("points ") + Colour + String(" ") + Shape; return S; }

BaseSeries* PointSeries::Clone() const
   { BaseSeries* bs = new PointSeries(*this); CheckSpace(bs); return bs; }

String LinePointSeries::Style() const
   { String S = String("linespoints ") + Colour + String(" ") + Shape; return S; }

BaseSeries* LinePointSeries::Clone() const
   { BaseSeries* bs = new LinePointSeries(*this); CheckSpace(bs); return bs; }

String DotSeries::Style() const
   { String S = String("dots ") + Colour; return S; }

BaseSeries* DotSeries::Clone() const
   { BaseSeries* bs = new DotSeries(*this); CheckSpace(bs); return bs; }

String YErrorBarSeries::Style() const
{
      String S = String("yerrorbars ") + Colour + String(" ") + Shape;
      return S;
}

BaseSeries* YErrorBarSeries::Clone() const
   { BaseSeries* bs = new YErrorBarSeries(*this); CheckSpace(bs); return bs; }

String XErrorBarSeries::Style() const
{
      String S = String("xerrorbars ") + Colour + String(" ") + Shape;
      return S;
}

BaseSeries* XErrorBarSeries::Clone() const
   { BaseSeries* bs = new XErrorBarSeries(*this); CheckSpace(bs); return bs; }

String XYErrorBarSeries::Style() const
{
      String S = String("xyerrorbars ") + Colour + String(" ") + Shape;
      return S;
}

BaseSeries* XYErrorBarSeries::Clone() const
   { BaseSeries* bs = new XYErrorBarSeries(*this); CheckSpace(bs); return bs; }

LL_Series::LL_Series(LL_Series* next, const BaseSeries& bs)
   : Next(next), Series(bs.Clone()) {}

LL_Series::~LL_Series()
   { if (Series) delete Series;  if (Next) delete Next; }

Line::Line(Real x1, Real y1, Real x2, Real y2)
   { Location.ReSize(2,2); Location << x1 << y1 << x2 << y2; }

Line::Line(Real x1, Real y1, Real z1, Real x2, Real y2, Real z2)
   { Location.ReSize(2,3); Location << x1 << y1 << z1 << x2 << y2 << z2; }

Arrow::Arrow(Real x1, Real y1, Real x2, Real y2)
   { Location.ReSize(2,2); Location << x1 << y1 << x2 << y2; }

Arrow::Arrow(Real x1, Real y1, Real z1, Real x2, Real y2, Real z2)
   { Location.ReSize(2,3); Location << x1 << y1 << z1 << x2 << y2 << z2; }

Label::Label(const String& label, Real x, Real y, Justification j)
      : TheLabel(label), justify(j)
   { Location.ReSize(1,2); Location << x << y; }

Label::Label(const String& label, Real x, Real y, Real z, Justification j)
      : TheLabel(label), justify(j)
   { Location.ReSize(1,3); Location << x << y << z; }

/// Add a line to a plot.
void Line::PlotIt(Plot2& P) const
{
   P.PartCommand("set arrow from ");
   int n = Location.Ncols(); int i;
   P.PartCommand(ToString(Location(1,1)));
   for (i = 2; i <= n; i++)
      { P.PartCommand(","); P.PartCommand(ToString(Location(1,i))); }
   P.PartCommand(" to ");
   P.PartCommand(ToString(Location(2,1)));
   for (i = 2; i <= n; i++)
      { P.PartCommand(","); P.PartCommand(ToString(Location(2,i))); }
   P.PartCommand(" nohead\n");
}

/// Add an arrow to a plot.
void Arrow::PlotIt(Plot2& P) const
{
   P.PartCommand("set arrow from ");
   int n = Location.Ncols(); int i;
   P.PartCommand(ToString(Location(1,1)));
   for (i = 2; i <= n; i++)
      { P.PartCommand(","); P.PartCommand(ToString(Location(1,i))); }
   P.PartCommand(" to ");
   P.PartCommand(ToString(Location(2,1)));
   for (i = 2; i <= n; i++)
      { P.PartCommand(","); P.PartCommand(ToString(Location(2,i))); }
   P.PartCommand(" head\n");
}

/// Add a label to a plot.
void Label::PlotIt(Plot2& P) const
{
   P.PartCommand("set label \"");
   P.PartCommand(TheLabel);
   P.PartCommand("\" at ");
   int n = Location.Ncols(); int i;
   P.PartCommand(ToString(Location(1,1)));
   for (i = 2; i <= n; i++)
      { P.PartCommand(","); P.PartCommand(ToString(Location(1,i))); }
   switch (justify)
   {
   case left: P.PartCommand(" left\n"); break;
   case right: P.PartCommand(" right\n"); break;
   case centre: case center: P.PartCommand(" center\n"); break;
   default: P.PartCommand("\n"); break;
   }
}



/// Add a series to a two dimensional plot.
Plot2& Plot2::operator<< (const BaseSeries& BS)
{
   if (BS.HasValues()) { LLS = new LL_Series(LLS,BS); CheckSpace(LLS); }
   return *this;
}

/// Add a function to a two dimensional plot
void Plot2::AddFunction(const String& function)
{
   if (Functions != "") Functions += ",";
   Functions += function;
   Functions += " \\\n";              // continuation character and newline
}

/// Actually plot a two dimensional plot.
void Plot2::DoPlot()
{
   ofstream fout; setup_file(CommandFile, fout);

   fout << "set title \"" << Title << "\"" << endl;

   // set the axes labels
   if (X_Axis.IsSet && X_Axis.Title != "")
      fout << "set xlabel \"" << X_Axis.Title << "\"" << endl;
   else fout << "set xlabel" << endl;

   if (Y_Axis.IsSet && Y_Axis.Title != "")
      fout << "set ylabel \"" << Y_Axis.Title << "\"" << endl;
   else fout << "set ylabel" << endl;

   // set the axes scale type (linear or log)

   fout << "set nologscale" << endl;
   if (Y_Axis.IsSet && Y_Axis.IsLog) fout << "set logscale y" << endl;
   if (X_Axis.IsSet && X_Axis.IsLog) fout << "set logscale x" << endl;

   // do we want a zero-axis

   fout << "set nozeroaxis" << endl;
   if (Y_Axis.IsSet && Y_Axis.ZeroAxis) fout << "set yzeroaxis" << endl;
   if (X_Axis.IsSet && X_Axis.ZeroAxis) fout << "set xzeroaxis" << endl;

    // do we want the key (legend)
   if (WK) fout << "set key spacing " << KeySpacing << endl;
   else fout << "set nokey" << endl;

   // do axis ranges
   fout << "set autoscale" << endl;
   if (Y_Axis.IsSet && Y_Axis.Wrange)
   {
      fout << "set yrange [" << Y_Axis.Min << ":" << Y_Axis.Max << "]" << endl;
      fout << "Y_min = " << Y_Axis.Min << endl;
      fout << "Y_max = " << Y_Axis.Max << endl;
   }
   if (X_Axis.IsSet && X_Axis.Wrange)
   {
      fout << "set xrange [" << X_Axis.Min << ":" << X_Axis.Max << "]" << endl;
      fout << "X_min = " << X_Axis.Min << endl;
      fout << "X_max = " << X_Axis.Max << endl;
   }

   // do command strings
   fout << Commands;

   bool first = true;
   fout << "plot \\" << endl;
   if (Functions != "") { first = false; fout << Functions; }
   for (LL_Series* lls = LLS; lls; lls = lls->Next)
   {
      if (first) first = false;
      else fout << ", ";
      fout << "\"" << TemporaryDirectory << bs << lls->Series->FN();
      fout << "\" title \"" << lls->Series->T() << "\" ";
      fout << "with  " << lls->Series->Style() << " \\" << endl;
                                 // "\\" is the line continuation code
   }
   fout << endl;
   if (WI)
   {
      fout << "set mouse nodoubleclick" << endl;
      fout << "COUNT = 0" << endl;
      fout << "MOUSE_X = -1E50" << endl;
      fout << "MOUSE_Y = -1E50" << endl;
      fout << "OLD_MOUSE_X = -1E50" << endl;
      fout << "OLD_MOUSE_Y = -1E50" << endl;
      fout << "set print '" << (TemporaryDirectory+bs) << "points.txt'" << endl;
      fout << "load '" << (TemporaryDirectory+bs) << "reread.txt'" << endl;
   }
   

   write_out(CommandFile, fout);

}

///Plot2 (and derived classes) destructor.
///Delete the whole linked list of graphs
Plot2::~Plot2()
{ if (LLS) delete LLS; }

/// Actually plot a three dimensional plot.
void Plot3::DoPlot()
{
   ofstream fout; setup_file(CommandFile, fout);
   
   fout << "set title \"" << Title << "\"" << endl;
   fout << "set parametric" << endl;

   // set the axes labels
   if (X_Axis.IsSet && X_Axis.Title != "")
      fout << "set xlabel \"" << X_Axis.Title << "\"" << endl;
   else fout << "set xlabel" << endl;

   if (Y_Axis.IsSet && Y_Axis.Title != "")
      fout << "set ylabel \"" << Y_Axis.Title << "\"" << endl;
   else fout << "set ylabel" << endl;

   if (Z_Axis.IsSet && Z_Axis.Title != "")
      fout << "set zlabel \"" << Z_Axis.Title << "\"" << endl;
   else fout << "set zlabel" << endl;

   // set the axes scale type (linear or log)

   fout << "set nologscale" << endl;
   if (Z_Axis.IsSet && Z_Axis.IsLog) fout << "set logscale z" << endl;
   if (Y_Axis.IsSet && Y_Axis.IsLog) fout << "set logscale y" << endl;
   if (X_Axis.IsSet && X_Axis.IsLog) fout << "set logscale x" << endl;

    // do we want the key (legend)
   if (WK) fout << "set key spacing " << KeySpacing << endl;
   else fout << "set nokey" << endl;

   // do axis ranges
   fout << "set autoscale" << endl;
   if (Z_Axis.IsSet && Z_Axis.Wrange)
      fout << "set zrange [" << Z_Axis.Min << ":" << Z_Axis.Max << "]"
	 << endl;
   if (Y_Axis.IsSet && Y_Axis.Wrange)
      fout << "set yrange [" << Y_Axis.Min << ":" << Y_Axis.Max << "]"
	 << endl;
   if (X_Axis.IsSet && X_Axis.Wrange)
      fout << "set xrange [" << X_Axis.Min << ":" << X_Axis.Max << "]"
	 << endl;

   // do command strings
   fout << Commands;

   bool first = true;
   fout << "splot \\" << endl;
   if (Functions != "") { first = false; fout << Functions; }
   for (LL_Series* lls = LLS; lls; lls = lls->Next)
   {
      if (first) first = false;
      else fout << ", ";
      fout << "\"" << TemporaryDirectory << bs << lls->Series->FN();
      fout << "\" title \"" << lls->Series->T() << "\" ";
      fout << "with  " << lls->Series->Style() << " \\" << endl;
                                 // "\\" is the line continuation code
   }
   fout << endl;


   write_out(CommandFile, fout);

}

/// Default constructor for MultiPlot
MultiPlot::MultiPlot()
   : Commands(""), Title("") {} 

/// Constructor for MultiPlot with title
MultiPlot::MultiPlot(const String& title)
   : Commands(""), Title(title) {} 

/// Set the size of a plot in a multiplot.
void MultiPlot::Size(Real x, Real y)
{
   Commands += "set size " + ToString(x) + ',' + ToString(y) + '\n';
}

/// Set the origin of a plot in a multiplot.
void MultiPlot::Origin(Real x, Real y)
{
   Commands += "set origin " + ToString(x) + ',' + ToString(y) + '\n';
}

/// Set the pointsize in a multiplot.
void MultiPlot::PointSize(Real ps)
{
   Commands += "set pointsize " + ToString(ps) + '\n';
}

/// Add a two dimensional plot to a multiplot.
MultiPlot& MultiPlot::operator<< (const Plot2& plot)
{
   Commands += "set title \"" + plot.Title + "\"\n";

   // set the axes labels
   if (plot.X_Axis.IsSet && plot.X_Axis.Title != "")
      Commands += "set xlabel \"" + plot.X_Axis.Title + "\"\n";
   else Commands += "set xlabel\n";

   if (plot.Y_Axis.IsSet && plot.Y_Axis.Title != "")
      Commands += "set ylabel \"" + plot.Y_Axis.Title + "\"\n";
   else Commands += "set ylabel\n";

   // set the axes scale type (linear or log)

   Commands += "set nologscale\n";
   if (plot.Y_Axis.IsSet && plot.Y_Axis.IsLog) Commands += "set logscale y\n";
   if (plot.X_Axis.IsSet && plot.X_Axis.IsLog) Commands += "set logscale x\n";

   // do we want a zero-axis

   Commands += "set nozeroaxis\n";
   if (plot.Y_Axis.IsSet && plot.Y_Axis.ZeroAxis) Commands += "set yzeroaxis\n";
   if (plot.X_Axis.IsSet && plot.X_Axis.ZeroAxis) Commands += "set xzeroaxis\n";

    // do we want the key (legend)
   if (plot.WK)
      Commands += ("set key spacing " + ToString(plot.KeySpacing) + "\n");
   else Commands += "set nokey\n";

   // do axis ranges
   Commands += "set autoscale\n";
   if (plot.Y_Axis.IsSet && plot.Y_Axis.Wrange)
      Commands += "set yrange [" + ToString(plot.Y_Axis.Min)
         + ":" + ToString(plot.Y_Axis.Max) + "]\n";
   if (plot.X_Axis.IsSet && plot.X_Axis.Wrange)
      Commands += "set xrange [" + ToString(plot.X_Axis.Min)
         + ":" + ToString(plot.X_Axis.Max) + "]\n";

   Commands += plot.Commands;

   bool first = true;
   Commands += "plot \\\n";
   if (plot.Functions != "") { first = false; Commands += plot.Functions; }
   for (LL_Series* lls = plot.LLS; lls; lls = lls->Next)
   {
      if (first) first = false;
      else Commands += ", ";
      Commands += "\"" + TemporaryDirectory + bs + lls->Series->FN();
      Commands += "\" title \"" + lls->Series->T() + "\" ";
      Commands += "with  " + lls->Series->Style() + " \\\n";
                                 // "\\" is the line continuation code
   }
   Commands += "\n";
   Commands += "reset";
   Commands += "\n";


   return *this;
}

/// Add a three dimensional plot to a multiplot.
MultiPlot& MultiPlot::operator<< (const Plot3& plot)
{
   Commands += "set title \"" + plot.Title + "\"\n";
   Commands += "set parametric\n";

   // set the axes labels
   if (plot.X_Axis.IsSet && plot.X_Axis.Title != "")
      Commands += "set xlabel \"" + plot.X_Axis.Title + "\"\n";
   else Commands += "set xlabel\n";

   if (plot.Y_Axis.IsSet && plot.Y_Axis.Title != "")
      Commands += "set ylabel \"" + plot.Y_Axis.Title + "\"\n";
   else Commands += "set ylabel\n";

   if (plot.Z_Axis.IsSet && plot.Z_Axis.Title != "")
      Commands += "set zlabel \"" + plot.Z_Axis.Title + "\"\n";
   else Commands += "set zlabel\n";

   // set the axes scale type (linear or log)

   Commands += "set nologscale\n";
   if (plot.Z_Axis.IsSet && plot.Z_Axis.IsLog) Commands += "set logscale z\n";
   if (plot.Y_Axis.IsSet && plot.Y_Axis.IsLog) Commands += "set logscale y\n";
   if (plot.X_Axis.IsSet && plot.X_Axis.IsLog) Commands += "set logscale x\n";


    // do we want the key (legend)
   if (plot.WK)
      Commands += ("set key spacing " + ToString(plot.KeySpacing) + "\n");
   else Commands += "set nokey\n";

   // do axis ranges
   Commands += "set autoscale\n";
   if (plot.Z_Axis.IsSet && plot.Z_Axis.Wrange)
      Commands += "set zrange [" + ToString(plot.Z_Axis.Min)
         + ":" + ToString(plot.Z_Axis.Max) + "]\n";
   if (plot.Y_Axis.IsSet && plot.Y_Axis.Wrange)
      Commands += "set yrange [" + ToString(plot.Y_Axis.Min)
         + ":" + ToString(plot.Y_Axis.Max) + "]\n";
   if (plot.X_Axis.IsSet && plot.X_Axis.Wrange)
      Commands += "set xrange [" + ToString(plot.X_Axis.Min)
         + ":" + ToString(plot.X_Axis.Max) + "]\n";

   Commands += plot.Commands;

   bool first = true;
   Commands += "splot \\\n";
   if (plot.Functions != "") { first = false; Commands += plot.Functions; }
   for (LL_Series* lls = plot.LLS; lls; lls = lls->Next)
   {
      if (first) first = false;
      else Commands += ", ";
      Commands += "\"" + TemporaryDirectory + bs + lls->Series->FN();
      Commands += "\" title \"" + lls->Series->T() + "\" ";
      Commands += "with  " + lls->Series->Style() + " \\\n";
                                 // "\\" is the line continuation code
   }
   Commands += "\n";
   Commands += "reset";
   Commands += "\n";

   return *this;
}

/// Actually plot the multiplot.
void MultiPlot::DoPlot()
{
   ofstream fout; setup_file(CommandFile, fout);

   // do command strings
   fout << "set origin 0,0" << endl;
   fout << "set size 1,1" << endl;
   fout << "set multiplot";
   if (Title != "") fout << " title \"" << Title << "\"";
   fout << endl;
   fout << Commands;
#ifdef set_unix_options
   fout << "set nomultiplot" << endl;
#endif
   //fout << "set multiplot" << endl;

   write_out(CommandFile, fout);

}


String TemporaryDirectory;
String CommandFile = "";

#ifdef use_namespace
}
#endif



// Additional documentation for gnuplot interface.
// Information that could have been included in gnuplot.h but is put here to
// avoid making gnuplot.h too complicated.

/// \fn BaseSeries::FN() const
/// \internal
/// Return the filename.

/// \fn BaseSeries::T() const
/// Return the title

/// \fn BaseSeries::Style() const
/// Return the string used by gnuplot to set the style.
/// \internal
/// The information required includes the style of the graph,
/// the colour, the shape (if relevant) etc.

/// \fn BaseSeries::Clone() const
/// Return a pointer to a copy of a graph.
/// \internal
/// This is used by LL_Series (linked list of Series) so we can destroy the original
/// series without ruining the plot. The data is on a file so Clone copies only the
/// filename, colour and shape information.

/// \fn BaseSeries::HasValues() const
/// \internal
/// Has any data been loaded into the graph.

/// \fn LineSeries::LineSeries(const String& title, const Matrix& data, const int colour)
/// See BaseSeries documentation.
/// data is nx2 for two dimensional plot and nx3 for three dimensional plot

/// \fn LineSeries::LineSeries(const String& title, const Matrix& data, const String& colour)
/// See BaseSeries documentation.
/// data is nx2 for two dimensional plot and nx3 for three dimensional plot

/// \fn ImpulseSeries::ImpulseSeries(const String& title, const Matrix& data, const int colour)
/// See BaseSeries documentation.
/// data is nx2 for two dimensional plot and nx3 for three dimensional plot

/// \fn ImpulseSeries::ImpulseSeries(const String& title, const Matrix& data, const String& colour)
/// See BaseSeries documentation.
/// data is nx2 for two dimensional plot and nx3 for three dimensional plot

/// \fn StepSeries::StepSeries(const String& title, const Matrix& data, const int colour)
/// See BaseSeries documentation.
/// data is nx2 for two dimensional plot (only 2D plot)

/// \fn StepSeries::StepSeries(const String& title, const Matrix& data, const String& colour)
/// See BaseSeries documentation.
/// data is nx2 for two dimensional plot (only 2D plot)

/// \fn BoxSeries::BoxSeries(const String& title, const Matrix& data, const int colour)
/// See BaseSeries documentation.
/// data is nx2 or nx3 for two dimensional plot (only 2D plot)

/// \fn BoxSeries::BoxSeries(const String& title, const Matrix& data, const String& colour)
/// See BaseSeries documentation.
/// data is nx2 or nx3 for two dimensional plot (only 2D plot)

/// \fn DotSeries::DotSeries(const String& title, const Matrix& data, const int colour)
/// See BaseSeries documentation.
/// data is nx2 for two dimensional plot and nx3 for three dimensional plot

/// \fn DotSeries::DotSeries(const String& title, const Matrix& data, const String& colour)
/// See BaseSeries documentation.
/// data is nx2 for two dimensional plot and nx3 for three dimensional plot

/// \fn PointSeries::PointSeries(const String& title, const Matrix& data, const int colour, const int shape)
/// See BaseSeries documentation.
/// data is nx2 for two dimensional plot and nx3 for three dimensional plot

/// \fn PointSeries::PointSeries(const String& title, const Matrix& data, const String& colour, const String& shape)
/// See BaseSeries documentation.
/// data is nx2 for two dimensional plot and nx3 for three dimensional plot

/// \fn LinePointSeries::LinePointSeries(const String& title, const Matrix& data, const int colour, const int shape)
/// See BaseSeries documentation.
/// data is nx2 for two dimensional plot and nx3 for three dimensional plot

/// \fn LinePointSeries::LinePointSeries(const String& title, const Matrix& data, const String& colour, const String& shape)
/// See BaseSeries documentation.
/// data is nx2 for two dimensional plot and nx3 for three dimensional plot

/// \fn YErrorBarSeries::YErrorBarSeries(const String& title, const Matrix& data, const int colour, const int shape)
/// See BaseSeries documentation.
/// data is nx3 or nx4 for two dimensional plot (only 2D plot)

/// \fn YErrorBarSeries::YErrorBarSeries(const String& title, const Matrix& data, const String& colour, const String& shape)
/// See BaseSeries documentation.
/// data is nx3 or nx4 for two dimensional plot (only 2D plot)

/// \fn XErrorBarSeries::XErrorBarSeries(const String& title, const Matrix& data, const int colour, const int shape)
/// See BaseSeries documentation.
/// data is nx3 or nx4 for two dimensional plot (only 2D plot)

/// \fn XErrorBarSeries::XErrorBarSeries(const String& title, const Matrix& data, const String& colour, const String& shape)
/// See BaseSeries documentation
/// data is nx3 or nx4 for two dimensional plot (only 2D plot)

/// \fn XYErrorBarSeries::XYErrorBarSeries(const String& title, const Matrix& data, const int colour, const int shape)
/// See BaseSeries documentation.
/// data is nx4 or nx6 for two dimensional plot (only 2D plot)

/// \fn XYErrorBarSeries::XYErrorBarSeries(const String& title, const Matrix& data, const String& colour, const String& shape)
/// See BaseSeries documentation.
/// data is nx4 or nx6 for two dimensional plot (only 2D plot)


///@}


