// value with derivatives - reverse mode

#ifndef RVWD_LIB
#define RVWD_LIB

#include "vwd.h"

class TapeElement
{
public:
   int location;                  // pointer to another variable on tape
   Real weight;                   // weight value associated with that variable
   Real cum_weight;               // used by reverse sweep process
   int ref_count;                 // how many RVWDs refer to this element
   bool first;                    // first element in an operation
   TapeElement() : cum_weight(0.0), ref_count(0) {}
};

// the real data for the TAPE - not accessed by user

class TAPE_Class
{
   long RefCount;
   void IncrRef() { RefCount++; }
   void DecrRef();
   ParameterSet PS;               // what we are wanting derivatives wrt
   TapeElement* TE;               // array of tape elements
   Matrix RV;                     // matrix containing derivatives as rows
   int N_max;                     // length of array of tape elements
   int N;                         // where we are now
   int N_largest;                 // the largest N has ever been
   int M_max;                     // length of array of RowVectors
   int M;                         // where we are now
   TAPE_Class(const ParameterSet& ps, int n, int m);
   ~TAPE_Class();
   void push(int l, Real w, bool f = false); // add new element
   RowVector reverse_sweep(int loc);         // evaluate the derivative
   void reset();                             // delete contents of tape
   void dump() const;                        // print contents of tape
   void purge();                             // recover obsolete space
   void purge(RVWD& rvwd);
   int FindSpareSlot();                      // look for spare slot in RV
   int IncludeDerivatives(const RowVector& rv);
   int Get_N_largest() const;     // the largest N has been
   friend class RVWD;
   friend class TAPE;
};

// what the user sees

class TAPE
{
   TAPE_Class* TC;
public:
   TAPE(const ParameterSet& ps, int n, int m);
   ~TAPE();
   TAPE(const TAPE& t) : TC(t.TC) { TC->IncrRef(); }
   void operator=(const TAPE& t);
   void push(int l, Real w, bool f)          // add new element
      { TC->push(l,w,f); }
   void push(int l, Real w)                  // add new element
      { TC->push(l,w); }
   RowVector reverse_sweep(int loc)          // evaluate the derivative
      { return TC->reverse_sweep(loc); }
   void reset() { TC->reset(); }             // delete contents of tape
   int Get_N() const { return TC->N; }
   int Get_N_largest() const { return TC->Get_N_largest(); }
   int Get_M() const { return TC->M; }
   ParameterSet GetParameterSet() const { return TC->PS; }
   void dump() const { TC->dump(); }         // print contents of tape
   void purge() { TC -> purge(); }           // recover obsolete space
   void purge(RVWD& rvwd) { TC->purge(rvwd); }
   int IncludeDerivatives(const RowVector& rv)
      { return TC->IncludeDerivatives(rv); }
   friend void AssertEqual(const TAPE& t1, const TAPE& t2);
   friend class RVWD;
};

class RVWD
{
   TAPE Tape;
   Real Value;
   int location;
   RVWD(int l, const TAPE& t, Real v);
   RVWD(const RVWD& rvwd, Real v, Real w);
   RVWD(const RVWD& rvwd1, const RVWD& rvwd2, Real v, Real w1, Real w2);
   void SetLocation();
   void UpdateLocation();
   void UpdateLocation(const TAPE& t);
   void UpdateLocation(const TAPE& t, int l);
   static TAPE TAPE_null;                     // for default constructor
   static ParameterSet PS_null;               // for default constructor
public:
   RVWD();
   explicit RVWD(Real v);
   RVWD(const RVWD&);
   RVWD(const TAPE& t, const VWD&);
   RVWD(const TAPE& t, Real v, int k);
   RVWD(const TAPE& t, Real v, const String& name);
   RVWD(const TAPE& t, Real v);                // set RVWD = constant
   ~RVWD() { --(Tape.TC->TE[location].ref_count); }
   Real GetValue() const { return Value; }
   ReturnMatrix GetDerivatives() const;
   TAPE GetTape() const { return Tape; }
   int GetLocation() const { return location; }
   ParameterSet GetParameterSet() const { return Tape.GetParameterSet(); }
   void operator=(const RVWD&);
   void operator+=(const RVWD&);
   void operator-=(const RVWD&);
   void operator*=(const RVWD&);
   void operator/=(const RVWD&);
   void operator+=(Real r) { Value += r; }
   void operator-=(Real r) { Value -= r; }
   void operator*=(Real);
   void operator/=(Real);
   RVWD operator-() const;
   friend void purge(RVWD& rvwd) { rvwd.Tape.purge(rvwd); }
   friend RVWD operator+(const RVWD&, const RVWD&);
   friend RVWD operator-(const RVWD&, const RVWD&);
   friend RVWD operator*(const RVWD&, const RVWD&);
   friend RVWD operator/(const RVWD&, const RVWD&);
   friend RVWD pow(const RVWD&, const RVWD&);
   friend RVWD operator+(Real, const RVWD&);
   friend RVWD operator-(Real, const RVWD&);
   friend RVWD operator*(Real, const RVWD&);
   friend RVWD operator/(Real, const RVWD&);
   friend RVWD pow(Real, const RVWD&);
   friend RVWD operator+(const RVWD&, Real);
   friend RVWD operator-(const RVWD&, Real);
   friend RVWD operator*(const RVWD&, Real);
   friend RVWD operator/(const RVWD&, Real);
   friend RVWD pow(const RVWD&, Real);
   friend RVWD pow(const RVWD&, int);
   friend RVWD exp(const RVWD&);
   friend RVWD log(const RVWD&);
   friend RVWD sin(const RVWD&);
   friend RVWD cos(const RVWD&);
   friend RVWD tan(const RVWD&);
#ifdef WANT_ERF
   friend RVWD erf(const RVWD&);
#endif
   friend class TAPE_Class;
};

// prototype of RVWD which is a function of a Real

class RVWDOfReal
{
protected:
   Real x;                             // Current x value
   bool xSet;                          // true if a value assigned to x

public:
   virtual void Set(Real X) { x = X; xSet = true; }
                                       // set x
   RVWDOfReal() : xSet(false) {}
   virtual RVWD operator()() = 0;
                                       // function value at current x
                                       // set current x
   RVWD operator()(Real X) { Set(X); return operator()(); }
                                       // set x, return value
};

// Gaussian numerical integration
RVWD GaussianIntegration32(RVWDOfReal& function, Real Lower, Real Upper);




#endif


// body file: rvwd.cpp


